<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Route;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class PermissionSeeder extends Seeder
{
    public function run()
    {
        // -----------------------------
        // 1. Delete existing role-permission links and permissions safely
        // -----------------------------
        DB::table('model_has_permissions')->delete();
        DB::table('permissions')->delete();

        $this->command->info('Existing permissions cleared.');

        // -----------------------------
        // 2. Get all named routes
        // -----------------------------
        $routes = collect(Route::getRoutes())->map(function ($route) {
            return $route->getName();
        })->filter()->unique();

        // -----------------------------
        // 3. Filter routes to include only meaningful permissions
        // -----------------------------
        $allowedPatterns = ['index', 'edit', 'create', 'delete', 'view'];
        $filteredRoutes = $routes->filter(function ($name) use ($allowedPatterns) {
            foreach ($allowedPatterns as $pattern) {
                if (str_contains($name, $pattern)) {
                    return true;
                }
            }
            return false;
        });

        // -----------------------------
        // 4. Create permissions
        // -----------------------------
        $permissions = [];
        foreach ($filteredRoutes as $routeName) {
            // Optional: module = first part of route name
            $parts = explode('.', $routeName);
            $module = $parts[0] ?? 'general';

            $permission = Permission::firstOrCreate([
                'name' => $routeName,
                'guard_name' => 'web',
                // 'module' => $module, // uncomment if you added module column
            ]);

            $permissions[] = $permission->id;

            $this->command->info("Permission created: $routeName ($module)");
        }

        // -----------------------------
        // 5. Create Super Admin role if not exists
        // -----------------------------
        $superAdmin = Role::firstOrCreate(
            ['name' => 'Super Admin', 'guard_name' => 'web']
        );

        // -----------------------------
        // 6. Assign all permissions to Super Admin
        // -----------------------------
        $superAdmin->syncPermissions($filteredRoutes);

        $this->command->info('All permissions assigned to Super Admin successfully!');
    }
}
